function stackedPlot(varargin)
%STACKEDPLOT   Stacked linear plot.
%
% DESCRIPTION:
%       stackePlot produces a series of stacked linear plots of the rows in
%       data (a 2D matrix) against the vector x. The vector y defines the
%       y-axis label for each linear plot. The plot can be annotated in the
%       normal fashion after display.
%
%       Examples:
%           stackedPlot(rand(5, 100));
%           stackedPlot(0.1:0.1:10, rand(5, 100));
%           stackedPlot(0.1:0.1:10, {'a', 'b', 'c', 'd', 'e'}, rand(5, 100));
%
% USAGE:
%       stackedPlot(data)
%       stackedPlot(x, data)
%       stackedPlot(x, y, data)
%
% INPUTS:
%       x       - vector defining the x-axis values
%       y       - vector defining the y-axis labels used for each plot
%       data    - 2D matrix to plot
%
% ABOUT:
%       author      - Bradley Treeby
%       date        - 14th November 2011
%       last update - 15th November 2011
%
% This function is part of the k-Wave Toolbox (http://www.k-wave.org)
% Copyright (C) 2009, 2010, 2011 Bradley Treeby and Ben Cox
%
% See also plot

% This file is part of k-Wave. k-Wave is free software: you can
% redistribute it and/or modify it under the terms of the GNU Lesser
% General Public License as published by the Free Software Foundation,
% either version 3 of the License, or (at your option) any later version.
% 
% k-Wave is distributed in the hope that it will be useful, but WITHOUT ANY
% WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
% FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
% more details. 
% 
% You should have received a copy of the GNU Lesser General Public License
% along with k-Wave. If not, see <http://www.gnu.org/licenses/>.

% check the inputs and extract the data size
switch nargin
    case 1
        data = varargin{1};
        [Ny, Nx] = size(data);
        x = 1:Nx;
        y = 1:Ny;
    case 2
        x = varargin{1};
        data = varargin{2};
        [Ny, Nx] = size(data);
        y = 1:Ny;
    case 3
        x = varargin{1};
        y = varargin{2};
        data = varargin{3};
        [Ny, Nx] = size(data);
    otherwise
        error('Unknown number of inputs.');
end
        
% get the maximum value within the input data
mx = max(data(:));

% create a series of linear plots of each row in data offset by the twice
% the maximum value
plot(x, data + repmat(2*mx*(Ny:-1:1).', 1, Nx), 'k-');

% set the y-axis labels to y
try
    % use new name for YTickLabel
    set(gca, 'YTick', (1:Ny)*2*mx, 'YTickLabel', fliplr(y), 'YLim', [0, (Ny + 1)*2*mx]);
catch
    % use old name for YTickLabel
    set(gca, 'YTick', (1:Ny)*2*mx, 'YTickLabels', fliplr(y), 'YLim', [0, (Ny + 1)*2*mx]);
end